#####################################################################
# R-code: Section 10.2.1
# File: Linex-Procedure.r
# Coded by: Marcella Niglio
#
# Asymmetric Loss Functions 
#
# Reference:
# Niglio, M. (2007).
#   Multi-step forecasts from threshold ARMA models using asymmetric 
#   loss functions.
#   Statistical Methods & Applications, 16(3), 395-410.
#   DOI: 10.1007/s10260-007-0044-x.
#####################################################################
linex.loss <- function(aa, nn, var.e)
{
	# Generate data for the linex function	
	# aa: parameter a
	# nn: length of the simulated series
	# var.e: variance of the simulated data
	#
	ee             <- rnorm(nn, mean = 0, sd = sqrt(var.e))
	linex          <- exp(aa * ee) - aa * ee - 1
	matr.linex     <- matrix(0, ncol=2, nrow=length(ee))
	matr.linex[,1] <- ee
	matr.linex[,2] <- linex
	return(matr.linex)
}

linlin.loss <- function(aa,bb,nn,var.e)
{
# Generate data for the functions LIN-LIN and QUAD-QUAD	
# aa:    parameter a
# bb:    parameter b
# nn:    length of the simulated series
# var.e: variance of the simulated data

ee          <- rnorm(nn,mean=0,sd=sqrt(var.e))
matr.ll.qq  <- matrix(0,ncol=3,nrow=nn,dimnames=list(NULL,c("errors","QuadQuad","LinLin"))) # Error matrices
ii <- 1
for (ii in 1: nn){
  if (ee[ii]>0){
    lin  <- aa*abs(ee[ii])
    quad <- aa*(ee[ii])^2 }
  else{
    lin  <- bb*abs(ee[ii]) 
    quad <- bb*(ee[ii])^2 }
    matr.ll.qq[ii,] <- c(ee[ii],quad,lin)
}
return(matr.ll.qq)
}

data.to.plot <- function(mm,dati){
#
ll     <- length(dati[,1])
nuovaM <- matrix(0,ncol=2,nrow=ll)
ii     <- 1
cont   <- 1
for(ii in 1:ll){
  if (dati[ii,2]<=mm){
     nuovaM[cont,2] <- dati[ii,2]
     cont           <- cont+1
   }
}
return(nuovaM[1:(cont-1),])
}

# LINEX PREDICTOR
simula.SETARMA <- function(N,h,AR.first,MA.first,AR.second,MA.second,var.err.1,var.err.2,d,v.s)
{
# This procedure allows to simulate data from a SETARMA model with two regimes
#
# N:         series length;
# h:         lead time;
# AR.first:  vector of the AR coefficients of the first regime 
#            (intercept included as first element of the vector);
# MA.first:  vector othe the MA coefficients of the first regime;
# AR.second: vector of the AR coefficients of the second regime 
#            (intercept included as first element of the vector);
# MA.second: vector othe the MA coefficients of the second regime;
# var.err.1: errors variance first regime;
# var.err.2: errors variance second regime;
# d:         threshold delay;
# v.s:       threshold value.
#
### NOTE: the AR order of the two regimes MUST be the same
#         (include null coefficients if necessary);
### NOTE: the MA order of the two regimes MUST be the same 
#         (include null coefficients if necessary);
#
# burn-in: 100
#

serie.obs         <- vector("numeric",N+h+100)
serie.err         <- vector("numeric",N+h+100)
p                 <- length(AR.first)-1
qo                <- length(MA.first)
vect.coeff.1      <- c(AR.first,1,MA.first)
vect.coeff.2      <- c(AR.second,1,MA.second)
iniz              <- max(p,qo, d)
serie.obs[1:iniz] <- rnorm(iniz,0,1)
serie.err[1:iniz] <- rnorm(iniz,0,1)
i <- 1
for(i in (iniz+1):length(serie.obs)){
  if (serie.obs[i-d]>=v.s){
    serie.err[i] <- rnorm(1,0,var.err.1)
    serie.obs[i] <- t(vect.coeff.1)%*% c(1,serie.obs[(i-1):(i-p)],serie.err[i:(i-qo)])}
  else {
	serie.err[i]<-rnorm(1,0,var.err.2)
	( serie.obs[i]<-t(vect.coeff.2)%*%c(1,serie.obs[(i-1):(i-p)],serie.err[i:(i-qo)]))
  }
}
serie <- list(obs=serie.obs[101:(N+100)],err=serie.err[101:(N+100)],prev.exp=serie.obs[(N+100+1):(N+100+h)])
return(serie)
# serie is a list with the following objects:
# obs      = vector of simulated data;
# err      = vector of simulated errors; 
# prev.exp = data used for ex-post forecasts;
}
#

prev.SETARMA.onestep <- function(N,lista.dati,h=1,AR.first,MA.first,AR.second,MA.second,var.err.1,var.err.2,d,v.s,aa)
{
#
# Generates least squares and linex forecast one step ahead from a SETARMA model 
# the variances of the errors of the two regimes (var.err.1 and var.err.2) are assumed 
# to be equal
#
# N:          series length;
# lista.dati: output simula.SETARMA procedure
# h:          lead time (h=1)
# AR.first:   vector of the AR coefficients of the first regime 
#             (intercept included as first element of the vector);
# MA.first:   vector othe the MA coefficients of the first regime;
# AR.second:  vector of the AR coefficients of the second regime 
#             (intercept included as first element of the vector);
# MA.second:  vector othe the MA coefficients of the second regime;
# var.err.1:  errors variance first regime;
# var.err.2:  errors variance second regime;
# d:          threshold delay;
# v.s:        threshold value.
#

p           <- length(AR.first)-1 
qo          <- length(MA.first)
primo.for   <- vector("numeric",h)
secondo.for <- vector("numeric",h)
m.prev.tot  <- matrix(0,ncol=h,nrow=9,dimnames=list(c("oss","prev.LS","err.prev.LS","prob.trans","var.err.LS","prev.LX","err.prev.LX","err.gen","var.err.gen"), NULL))
# m.prev.tot is a matrix with nine rows:
# [1,]: ex-post data; 
# [2,]: LS forecasts; 
# [3,]: LS forecast errors; 
# [4,]: values of the indicator function/probability transition
# [5,]: variance of the LS forecast errors;
# [6,]: Linex forecasts; 
# [7,]: Linex forecast errors; 
# [8,]: generalized errors; 
# [9,]: variance generalized errors
#

m.prev.tot[1,] <- lista.dati$prev.exp
pr.emp         <- 1  #starting value empirical probability
ind            <- as.matrix(seq(from=1,to=(qo+1),by=1))
matr.trian     <- apply(ind,1,FUN=function(ii,qqo=qo+1){c(rep(1,ii-1),rep(0,qqo-ii+1))})
vett.temp      <- c(0, lista.dati$err[N:(N-qo+1)])
matr.prev      <- matrix(0,ncol=qo+1,nrow=h)
i <- 1
for(i in 1:(h)){
   if (i <=(qo)){
	matr.prev[i,] <- matr.trian[i,]*vett.temp[1:(qo+1)]
	vett.temp     <- c(0,vett.temp)}
   else
        matr.prev[i,] <- 0
		}
vect.coeff.1 <- c(AR.first, 1, MA.first)
vect.coeff.2 <- c(AR.second, 1, MA.second)
val.oss      <- lista.dati$obs[N:(N-20)]
psi.list     <- pesi(h,d, AR.first, MA.first, AR.second, MA.second) 
i <- 1
vett.prev      <- c(1, val.oss[1:p], matr.prev[i,])
primo.for[i]   <- vect.coeff.1%*%vett.prev
secondo.for[i] <- vect.coeff.2%*%vett.prev
  if(val.oss[d]>=v.s){				
	# CASE h<=d
	forec           <- primo.for[i]
	m.prev.tot[4,i] <- 1
	m.prev.tot[5,i] <- var.err.1*sum(psi.list$psi1[1:i]^2)
	var.cond        <- var.err.1
	m.prev.tot[2,i] <- forec 
	m.prev.tot[3,i] <- m.prev.tot[1,i]-forec  
	m.prev.tot[6,i] <- forec+aa/2*var.cond
	m.prev.tot[7,i] <- m.prev.tot[1, i]-m.prev.tot[6,i]
	m.prev.tot[8,i] <- aa-aa*exp(aa*m.prev.tot[7,i])
	m.prev.tot[9,i] <- aa^2*(exp(aa^2*var.cond)-1)
	}
   if(val.oss[d]<v.s){
	forec           <- secondo.for[i]
	m.prev.tot[5,i] <- var.err.2*sum(psi.list$psi2[1:i]^2)
	var.cond        <- var.err.1
	m.prev.tot[2,i] <- forec 
	m.prev.tot[3,i] <- m.prev.tot[1,i]-forec  
	m.prev.tot[6,i] <- forec+aa/2*var.cond
	m.prev.tot[7,i] <- m.prev.tot[1, i]-m.prev.tot[6,i]
	m.prev.tot[8,i] <- aa-aa*exp(aa*m.prev.tot[7,i])
	m.prev.tot[9,i] <- aa^2*(exp(aa^2*var.cond)-1)
	}
return(list(MatricePrevisioni=m.prev.tot,prev.primo.regime=primo.for,prev.secondo.regime=secondo.for))
# The list returned has the following objects:
# MatricePrevisioni:   matrix with forecasts
# prev.primo.regime:   first regime forecasts
# prev.secondo.regime: second regime forecasts
}
#

prev.SETARMA.twosteps <- function(N,lista.dati,h,AR.first,MA.first,AR.second,MA.second,var.err.1,var.err.2,d,v.s,aa)
{
#
# Generates least squares and linex forecast two steps ahead from a SETARMA model 
# the variances of the errors of the two regimes (var.err.1 and var.err.2) are 
# assumed to be equal
#
# N: series length;
# lista.dati: output simula.SETARMA procedure
# h:          lead time (h=1 or h=2)
# AR.first:   vector of the AR coefficients of the first regime (intercept included as first element of the vector);
# MA.first:   vector othe the MA coefficients of the first regime;
# AR.second:  vector of the AR coefficients of the second regime (intercept included as first element of the vector);
# MA.second:  vector othe the MA coefficients of the second regime;
# var.err.1:  errors variance first regime;
# var.err.2:  errors variance second regime;
# d:          threshold delay;
# v.s:        threshold value.
#

p           <- length(AR.first)-1
qo          <- length(MA.first)
primo.for   <- vector("numeric",h)
secondo.for <- vector("numeric",h)
m.prev.tot  <- matrix(0,ncol=h, nrow=9,dimnames=list(c("oss","prev.LS","err.prev.LS","prob.trans","var.err.LS","prev.LX","err.prev.LX","err.gen","var.err.gen"),NULL))
# m.prev.tot is a matrix with nine rows:
# [1,]: ex-post data; 
# [2,]: LS forecasts; 
# [3,]: LS forecast errors; 
# [4,]: values of the indicator function/probability transition
# [5,]: variance of the LS forecast errors;
# [6,]: Linex forecasts; 
# [7,]: Linex forecast errors; 
# [8,]: generalized errors; 
# [9,]: variance generalized errors
#
m.prev.tot[1,] <- lista.dati$prev.exp
pr.emp         <- 1
ind            <- as.matrix(seq(from=1,to=(qo+1),by=1))
matr.trian     <- apply(ind,1,FUN=function(ii,qqo=qo+1){c(rep(1,ii-1),rep(0,qqo-ii+1))})
vett.temp      <- c(0, lista.dati$err[N:(N-qo+1)])
matr.prev      <- matrix(0,ncol=qo+1,nrow=h)

i <- 1
for(i in 1:(h)){
  if (i <=(qo)){
    matr.prev[i,] <- matr.trian[i,]*vett.temp[1:(qo+1)]
    vett.temp     <- c(0,vett.temp)}
  else
    matr.prev[i,] <- 0
  }
  
vect.coeff.1 <- c(AR.first,1,MA.first)
vect.coeff.2 <- c(AR.second,1,MA.second)
val.oss      <- lista.dati$obs[N:(N-20)]
psi.list     <- pesi(h,d,AR.first,MA.first,AR.second,MA.second)  # Computing psi's
i<-1
for(i in 1:h){
   vett.prev      <- c(1,val.oss[1:p],matr.prev[i,])
   primo.for[i]   <- vect.coeff.1%*%vett.prev
   secondo.for[i] <- vect.coeff.2%*%vett.prev
   if(val.oss[d]>=v.s){				
	# CASE h<=d
	forec           <- primo.for[i]
	m.prev.tot[4,i] <- 1
	m.prev.tot[5,i] <- var.err.1*sum(psi.list$psi1[1:i]^2)
	if (i==1) 
	  var.cond <- var.err.1
	else (var.cond<-var.err.1*((AR.first[2]-MA.first[1])^2+1))
	  m.prev.tot[2,i] <- forec 
	  m.prev.tot[3,i] <- m.prev.tot[1,i]-forec
	  m.prev.tot[6,i] <- forec+aa/2*var.cond
	  m.prev.tot[7,i] <- m.prev.tot[1, i]-m.prev.tot[6,i]
	  m.prev.tot[8,i] <- aa-aa*exp(aa*m.prev.tot[7,i])
	  m.prev.tot[9,i] <- aa^2*(exp(aa^2*var.cond)-1)
	}
	if(val.oss[d]<v.s){
		forec <- secondo.for[i]
		m.prev.tot[5,i] <- var.err.2*sum(psi.list$psi2[1:i]^2)
		if (i==1) 
		  var.cond <- var.err.1
		else (var.cond<-var.err.1*((AR.second[2]-MA.second[1])^2+1))
		  m.prev.tot[2,i] <- forec 
		  m.prev.tot[3,i] <- m.prev.tot[1,i]-forec
		  m.prev.tot[6,i] <- forec+aa/2*var.cond
		  m.prev.tot[7,i] <- m.prev.tot[1, i]-m.prev.tot[6,i]
		  m.prev.tot[8,i] <- aa-aa*exp(aa*m.prev.tot[7,i])
		  m.prev.tot[9,i] <- aa^2*(exp(aa^2*var.cond)-1)
	}
	val.oss <- c(forec,val.oss)
}
return(list(MatricePrevisioni=m.prev.tot,prev.primo.regime=primo.for,prev.secondo.regime=secondo.for))
# The list returned has the following objects:
#   MatricePrevisioni:   matrix with forecasts (see the content of m.prev.tot)
#   prev.primo.regime:   first regime forecasts
#   prev.secondo.regime: second regime forecasts
}
#

pesi <- function(h,d,AR.first,MA.first,AR.second,MA.second){
#
# h:          lead time;
# d:          threshold delay;
# AR.first:   vector of the AR coefficients of the first regime 
#             (intercept included as first element of the vector);
# MA.first:   vector of the the MA coefficients of the first regime;
# AR.second:  vector of the AR coefficients of the second regime 
#             (intercept included as first element of the vector);
# MA.second:  vector of the the MA coefficients of the second regime;
#
# This procedure allows to obtain the weights to estimate the errors variance
#

h5     <- 5*h
psi.1  <- vector("numeric",(h5 ))
psi.2  <- vector("numeric",(h5 ))
dimens <- h5  - length(AR.first)
if(dimens > 1) rip <- rep(0, dimens) else rip <- 0
matr.coeff.1 <- matrix(c(AR.first[2:length(AR.first)],0,rip,MA.first,rip),ncol=(length(AR.first)+length(rip)),nrow=2,byrow=T) 
matr.coeff.2 <- matrix(c(AR.second[2:length(AR.second)],0,rip,MA.second,rip),ncol=(length(AR.second)+length(rip)),nrow=2,byrow=T)
psi.temp.1   <- c(1)
psi.temp.2   <- c(1)

ii <- 1
for (ii in 2:(h5)){
   ll         <- length(psi.temp.1)
   t1         <- sum(psi.temp.1[1:ll]*matr.coeff.1[1,1:ll])-matr.coeff.1[2,(ii-1)]
   t2         <- sum(psi.temp.2[1:ll]*matr.coeff.2[1,1:ll])-matr.coeff.2[2,(ii-1)]
   psi.temp.1 <- c(t1,psi.temp.1)
   psi.temp.2 <- c(t2,psi.temp.2)}
   psi.1      <- psi.temp.1[length(psi.temp.1):1]
   psi.2      <- psi.temp.2[length(psi.temp.2):1]
return(list(psi1=psi.1,psi2=psi.2))
# psi1: first regime weigths
# psi2: second regime weigths
}
#

main.program <- function(N,h,AR.first,MA.first,AR.second,MA.second,d,v.s,aa)
{
# This function generates data from a SETARMA model and generalized prediction
# errors with their variances respectively
#	
# seed = 300
# N:         series length
# h:         lead time
# AR.first:  vector of the AR coefficients of the first regime 
#            (intercept included as first element of the vector);
# MA.first:  vector of the the MA coefficients of the first regime;
# AR.second: vector of the AR coefficients of the second regime 
#            (intercept included as first element of the vector);
# MA.second: vector of the the MA coefficients of the second regime;
# var.err.1: errors variance first regime;
# var.err.2: errors variance second regime;
# d:         threshold delay
# v.s:       threshold value
# aa:        linex function parameter
#
### NOTE: the AR order of the two regimes MUST be the same 
#         (include null coefficients if necessary);
### NOTE: the MA order of the two regimes MUST be the same 
#         (include null coefficients if necessary);
#
serie.sim <- simula.SETARMA(N,h,AR.first,MA.first,AR.second,MA.second,var.err.1,var.err.2,d,v.s)
mm.pp     <- prev.SETARMA.twosteps(N,serie.sim,h,AR.first,MA.first,AR.second,MA.second,var.err.1,var.err.2,d,v.s,aa)
return(mm.pp)}
